<?php
namespace ovaex_elementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use OVAEX_Settings;

if ( !defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ovaex_exhibition_ajax extends Widget_Base {

	public function get_name() {
		return 'ovaex_exhibition_ajax';
	}

	public function get_title() {
		return esc_html__( 'Exhibition Load Ajax', 'ovaex' );
	}

	public function get_icon() {
		return 'eicon-posts-grid';
	}

	public function get_categories() {
		return [ 'ovatheme_exhibition' ];
	}

	public function get_script_depends() {
		wp_enqueue_script( 'script-elementor-ovaex' );
		
		return [ '' ];
	}

	public function get_style_depends() {
		wp_enqueue_style( 'ovaex-style', OVAEX_PLUGIN_URI.'assets/css/frontend/ovaex-style.css', array(), null );
		return [];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'section_content',
			[
				'label' => esc_html__( 'Content', 'ovaex' ),
			]
		);

			$this->add_control(
				'total_count',
				[
					'label'   => esc_html__( 'Total Exhibition', 'ovaex' ),
					'type'    => Controls_Manager::NUMBER,
					'default' => 3
				]
			);

			$this->add_control(
				'categories',
				[
					'label'   	=> esc_html__( 'Categories', 'ovaex' ),
					'multiple' 	=> true,
					'type'    	=> \Elementor\Controls_Manager::SELECT2,
					'options' 	=> $this->get_all_categories(),
				]
			);

			$this->add_control(
				'artist',
				[
					'label'   	=> esc_html__( 'Artist', 'ovaex' ),
					'type'    	=> \Elementor\Controls_Manager::SELECT,
					'default' 	=> '',
					'options' 	=> $this->get_all_artists(),
				]
			);

			$this->add_control(
				'order_by',
				[
					'label'   => esc_html__( 'Order By', 'ovaex' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'title',
					'options' => [
						'title'                  => esc_html__( 'Title', 'ovaex' ),
						'exhibition_custom_sort' => esc_html__( 'Custom Sort', 'ovaex' ),
						'ex_start_date'          => esc_html__( 'Start Date', 'ovaex' ),
						'ID'                     => esc_html__( 'ID', 'ovaex' ),					
					],
				]
			);

			$this->add_control(
				'order',
				[
					'label'   => esc_html__( 'Order', 'ovaex' ),
					'type'    => Controls_Manager::SELECT,
					'default' => 'DESC',
					'options' => [
						'DESC' => esc_html__( 'Descending', 'ovaex' ),
						'ASC'  => esc_html__( 'Ascending', 'ovaex' ),
					],
				]
			);

			$this->add_control(
				'upcoming_day',
				[
					'label' 		=> esc_html__('Upcoming Day','ovaex'),
					'type' 			=> Controls_Manager::NUMBER,
					'default' 		=> 7,
					'min' 			=> 1,
					'description' 	=> esc_html__('Add a day before the event takes place, Ex: 7 day', 'ovaex')
				]
			);

			$this->add_control(
				'show_loadmore',
				[
					'label' => esc_html__( 'Show Load More', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

		$this->end_controls_section();

		// Filter
		$this->start_controls_section(
			'section_filter',
			[
				'label' => esc_html__( 'Filter', 'ovaex' ),
			]
		);

			$this->add_control(
				'default_filter',
				[
					'label' 	=> esc_html__('Default Filter','ovaex'),
					'type'  	=> Controls_Manager::SELECT,
					'default' 	=> 'current',
					'options' 	=> [
						'past' => esc_html__('Past','ovaex'),
						'current' => esc_html__('Current','ovaex'),
						'upcoming' => esc_html__('Upcoming','ovaex')
					]
				]
			);

			$this->add_control(
				'show_current',
				[
					'label' => esc_html__( 'Show Current', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_upcoming',
				[
					'label' => esc_html__( 'Show Upcoming', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_past',
				[
					'label' => esc_html__( 'Show Past', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

		$this->end_controls_section();

		// Results
		$this->start_controls_section(
			'section_results',
			[
				'label' => esc_html__( 'Results', 'ovaex' ),
			]
		);

			$this->add_control(
				'show_image',
				[
					'label' => esc_html__( 'Show Image', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_date',
				[
					'label' => esc_html__( 'Show Date', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_category',
				[
					'label' => esc_html__( 'Show Category', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_title',
				[
					'label' => esc_html__( 'Show Title', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_link_of_title',
				[
					'label' => esc_html__( 'Show Link of Title', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

			$this->add_control(
				'show_view_exhibition',
				[
					'label' => esc_html__( 'Show View Exhibition', 'ova-doc' ),
					'type' => \Elementor\Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Show', 'ova-doc' ),
					'label_off' => esc_html__( 'Hide', 'ova-doc' ),
					'return_value' => 'yes',
					'default' => 'yes',
				]
			);

		$this->end_controls_section();

		// Title style
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__( 'Title', 'ovaex' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'title_typography',
				'label' 	=> esc_html__( 'Typography', 'ovaex' ),
				'selector' 	=> '{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content h3',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' 	=> esc_html__( 'Title Color', 'ovaex' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content h3 a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_hover_color',
			[
				'label' 	=> esc_html__( 'Title Hover Color', 'ovaex' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content h3 a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_magin',
			[
				'label' 		=> esc_html__( 'Title Margin', 'ovaex' ),
				'type' 			=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', 'em', '%' ],
				'selectors' 	=> [
					'{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' 	=> 'before',
			]
		);

		$this->end_controls_section();

		// Date_Time_style
		$this->start_controls_section(
			'section_date_time_style',
			[
				'label' => esc_html__( 'Date/Time', 'ovaex' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'date_time_typography',
				'label' 	=> esc_html__( 'Typography', 'ovaex' ),
				'selector' 	=> '{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content .times span',
			]
		);

		$this->add_control(
			'date_time_color',
			[
				'label' 	=> esc_html__( 'Text Color', 'ovaex' ),
				'type' 		=> Controls_Manager::COLOR,
				'default' 	=> '',
				'selectors' => [
					'{{WRAPPER}} .exhibition_arc .wrap-content .post-items .content .times span' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		// load more
		$this->start_controls_section(
			'section_load_more_style',
			[
				'label' => esc_html__( 'Load More', 'ovaex' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'load_more_typography',
				'label' 	=> esc_html__( 'Typography', 'ovaex' ),
				'selector' 	=> '{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span',
			]
		);

		$this->add_responsive_control(
			'read_more_padding',
			[
				'label' 		=> esc_html__( 'Padding', 'ovaex' ),
				'type' 			=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', 'em', '%' ],
				'selectors' 	=> [
					'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' 	=> 'before',
			]
		);

		$this->add_responsive_control(
			'read_more_margin',
			[
				'label' 		=> esc_html__( 'Margin', 'ovaex' ),
				'type' 			=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', 'em', '%' ],
				'selectors' 	=> [
					'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' 	=> 'before',
			]
		);

		$this->start_controls_tabs( 'tabs_button_style' );
			// Tab button normal
			$this->start_controls_tab(
				'tab_button_normal',
				[
					'label' => esc_html__( 'Normal', 'ovaex' ),
				]
			);

			$this->add_control(
				'read_more_text_color',
				[
					'label' 	=> esc_html__( 'Text Color', 'ovaex' ),
					'type' 		=> Controls_Manager::COLOR,
					'default' 	=> '',
					'selectors' => [
						'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'read_more_background_color',
				[
					'label' 	=> esc_html__( 'Background Color', 'ovaex' ),
					'type' 		=> Controls_Manager::COLOR,
					'default' 	=> '',
					'selectors' => [
						'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span' => 'background-color: {{VALUE}};',
					],
				]
		);

		$this->end_controls_tab();

			// Tab button hover
			$this->start_controls_tab(
				'tab_button_hover',
				[
					'label' => esc_html__( 'Hover', 'ovaex' ),
				]
			);

			$this->add_control(
				'hover_color',
				[
					'label' 	=> esc_html__( 'Text Color', 'ovaex' ),
					'type' 		=> Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more:hover, {{WRAPPER}} .exhibition_arc .ova-nodata span:hover' => 'color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'button_background_hover_color',
				[
					'label' 	=> esc_html__( 'Background Color', 'ovaex' ),
					'type' 		=> Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more:hover, {{WRAPPER}} .exhibition_arc .ova-nodata span:hover' => 'background-color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'button_hover_border_color',
				[
					'label' 	=> esc_html__( 'Border Color', 'ovaex' ),
					'type' 		=> Controls_Manager::COLOR,
					'condition' => [
						'border_border!' => '',
					],
					'selectors' => [
						'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more:hover, {{WRAPPER}} .exhibition_arc .ova-nodata span:hover' => 'border-color: {{VALUE}};',
					],
				]
			);

			$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' 			=> 'border',
				'placeholder' 	=> '1px',
				'default' 		=> '1px',
				'selector' 		=> '{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span',
				'separator' 	=> 'before',
			]
		);

		$this->add_control(
			'border_radius',
			[
				'label' 		=> esc_html__( 'Border Radius', 'ovaex' ),
				'type' 			=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', '%' ],
				'selectors' 	=> [
					'{{WRAPPER}} .exhibition_arc .ova_more_post .load-more, {{WRAPPER}} .exhibition_arc .ova-nodata span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	private function get_all_categories() {
		$result = array();

		$args = array(
           'taxonomy' 	=> 'cat_exhibition',
           'orderby' 	=> 'name',
           'order'   	=> 'ASC'
       	);
	
		$categories = get_categories( $args );

		if ( $categories ) {
			foreach( $categories as $cate ) {
				$result[$cate->slug] = $cate->cat_name;
			}
		}

		return $result;
	}

	private function get_all_artists() {
		$result = array(
			'' => esc_html__( 'All', 'ovaex' ),
		);

		$args_artists = array(
			'post_type'      	=> 'artist',
			'post_status' 		=> 'publish',
			'posts_per_page' 	=> -1,
			'fields' 			=> 'ids',
		);

		$artists = get_posts( $args_artists );

		if ( !empty( $artists ) && is_array( $artists ) ) {
			foreach( $artists as $id ) {
				$result[$id] = get_the_title( $id );
			}
		}

		return $result;
	}

	protected function render() {
		$settings = $this->get_settings();

		$default_filter         = $settings['default_filter'];
		$upcoming_day        	= absint( $settings['upcoming_day'] );
		$ovaex_order_by      	= $settings['order_by'];
		$ovaex_order         	= $settings['order'];
		$categories 			= $settings['categories'];
		$artist 				= $settings['artist'];
		$ovaex_post_per_page 	= absint( $settings['total_count'] );
		$paged               	= ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

		switch ( $ovaex_order_by ) {
			case 'title' : {
				$agr_order = [
					'orderby'   => 'title',
					'order'     => $ovaex_order,
				];
				break;
			}
			case 'ID' : {
				$agr_order = [
					'orderby'   => 'ID',
					'order'  => $ovaex_order,
				];
				break;
			}
			case 'exhibition_custom_sort' : {
				$agr_order = [
					'orderby'   => 'meta_value_num',
					'meta_key' => 'exhibition_custom_sort',
					'order' => $ovaex_order,
				];
				break;
			}
			case 'ex_start_date' : {
				$agr_order = [
					'orderby'   => 'meta_value_num',
					'meta_key' => 'ex_start_date',
					'order' => $ovaex_order,
				];
				break;
			}
			default: {
				$agr_order = [
					'orderby'  => 'ID',
					'order'  => $ovaex_order,
				];
				break;
			}
		}

		if ( $default_filter == 'current' ) {
			$args_basic = array(
				'post_type'      => 'exhibition',
				'post_status'    => 'publish',
				'paged'          => $paged,
				'posts_per_page' => $ovaex_post_per_page,
				'meta_query'     => array(
					array(
						'relation' => 'OR',
						array(
							'key' 		=> 'ex_start_date',
							'value' 	=> array( current_time('timestamp' )-1, current_time('timestamp' )+(24*60*60)+1 ),
							'type' 		=> 'numeric',
							'compare' 	=> 'BETWEEN'	
						),
						array(
							'relation' => 'AND',
							array(
								'key' 		=> 'ex_start_date',
								'value' 	=> current_time('timestamp' ),
								'compare' 	=> '<',
								'type'		=> 'NUMERIC'
							),
							array(
								'key' 		=> 'ex_end_date',
								'value' 	=> current_time('timestamp' ),
								'compare' 	=> '>=',
								'type'		=> 'NUMERIC'
							)
						)
					)
				)
			);
		} elseif ( $default_filter == 'upcoming' ) {
			$args_basic = array(
				'post_type'      => 'exhibition',
				'post_status'    => 'publish',
				'paged'          => $paged,
				'posts_per_page' => $ovaex_post_per_page,
				'meta_query'     => array(
					array(
						'relation' => 'AND',
						array(
							'key' 		=> 'ex_start_date',
							'value' 	=> current_time( 'timestamp' ),
							'compare' 	=> '>',
							'type'		=> 'NUMERIC'
						),
						array(
							'key' 		=> 'ex_end_date',
							'value' 	=> current_time( 'timestamp') + ( $upcoming_day * 24 * 60 * 60 ),
							'compare' 	=> '<=',
							'type'		=> 'NUMERIC'
						)	
					)
				)
			);		
		} elseif ( $default_filter == 'past' ) {
			$args_basic = array(
				'post_type'      => 'exhibition',
				'post_status'    => 'publish',
				'paged'          => $paged,
				'posts_per_page' => $ovaex_post_per_page,
				
				'meta_query'   => array(
					array(
						'key'     	=> 'ex_end_date',
						'value'   	=> current_time( 'timestamp' ),
						'compare' 	=> '<',
						'type'		=> 'NUMERIC'
					),
				),
			);
		}

		$args_cat = $args_artist = array();

		if ( !empty( $categories ) && is_array( $categories ) ) {
			$args_cat = array(
				'tax_query' => array(
					array(
						'taxonomy' => 'cat_exhibition',
						'field'    => 'slug',
						'terms'    => $categories,
					)
				)
			);
		}

		if ( !empty( $artist ) ) {
			$args_artist = array(
				'meta_query' => array(
					array(
						'key'     => 'exhibition_artist',
						'value'   => $artist,
						'compare' => 'LIKE'
					)
				)
			);
		}

		$args 			= array_merge_recursive( $args_basic, $agr_order, $args_cat, $args_artist );
		$exhibition  	= new \WP_Query( $args );

		// show load more button
		$show_loadmore = $settings['show_loadmore'];

		// show filter
		$show_past     = $settings['show_past'];
		$show_current  = $settings['show_current'];
		$show_upcoming = $settings['show_upcoming'];

		// show fields
		$args_show = array(
			'show_title' 		   => $settings['show_title'],
			'show_link_of_title'   => $settings['show_link_of_title'],
			'show_image'   		   => $settings['show_image'],
			'show_date'			   => $settings['show_date'],
			'show_category'	       => $settings['show_category'],
			'show_view_exhibition' => $settings['show_view_exhibition'],
		);

		?>

		<div class="ova-exhibition-ajax exhibition_arc" data-args_show="<?php echo esc_attr( json_encode($args_show) );?>">
			<div class="ova-wrap-exhibition wrap-content element-arc">

				<?php if($show_past == 'yes' || $show_current == 'yes' || $show_upcoming == 'yes') : ?>
					<ul class="nav nav-tabs">
						<?php if($show_current == 'yes') : ?>
						<li class="nav-item exhibition_tab_item">
							<a class="nav-link <?php if($default_filter == 'current') echo 'active';?>" 
								href="#current" 
								role="tab" 
								data-toggle="tab" 
								data-behav="tab" 
								data-type_action="current" 
								data-post="exhibition" 
								data-perpage="<?php echo esc_attr( $ovaex_post_per_page ); ?>" 
								data-paged="1" 
								data-order-by="<?php echo esc_attr( $ovaex_order_by );?>" 
								data-order="<?php echo esc_attr( $ovaex_order ); ?>" 
								data-upcoming="<?php echo esc_attr( $upcoming_day ); ?>" 
								data-categories="<?php echo esc_attr( json_encode( $categories ) ); ?>" 
								data-artist="<?php echo esc_attr( $artist ); ?>">
								<?php esc_html_e( 'current','ovaex' ); ?>
							</a>
						</li>
						<?php endif;?>

						<?php if($show_upcoming == 'yes') : ?>
						<li class="nav-item exhibition_tab_item">
							<a class="nav-link <?php if($default_filter == 'upcoming') echo 'active';?>" 
								href="#" 
								role="tab" 
								data-paged="1" 
								data-toggle="tab" 
								data-behav="tab" 
								data-type_action="upcoming" 
								data-post="exhibition" 
								data-perpage="<?php echo esc_attr( $ovaex_post_per_page ); ?>" 
								data-order-by="<?php echo esc_attr( $ovaex_order_by ); ?>" 
								data-order="<?php echo esc_attr( $ovaex_order ); ?>" 
								data-upcoming="<?php echo esc_attr( $upcoming_day ); ?>" 
								data-categories="<?php echo esc_attr( json_encode( $categories ) ); ?>" 
								data-artist="<?php echo esc_attr( $artist ); ?>">
								<?php esc_html_e( 'upcoming','ovaex' ); ?>
							</a>
						</li>
						<?php endif;?>

						<?php if($show_past == 'yes') : ?>
						<li class="nav-item exhibition_tab_item">
							<a class="nav-link <?php if($default_filter == 'past') echo 'active';?>" 
								href="#past" 
								role="tab" 
								data-paged="1" 
								data-behav="tab" 
								data-toggle="tab" 
								data-type_action="past" 
								data-post="exhibition" 
								data-perpage="<?php echo esc_attr( $ovaex_post_per_page ); ?>" 
								data-order-by="<?php echo esc_attr( $ovaex_order_by ); ?>" 
								data-order="<?php echo esc_attr( $ovaex_order ); ?>" 
								data-upcoming="<?php echo esc_attr( $upcoming_day ); ?>" 
								data-categories="<?php echo esc_attr( json_encode( $categories ) ); ?>" 
								data-artist="<?php echo esc_attr( $artist ); ?>">
								<?php esc_html_e( 'past','ovaex' ); ?>
							</a>
						</li>
						<?php endif;?>
					</ul>
				<?php endif; ?>

				<div class="tab-content">
					<?php if ( $exhibition->have_posts() ) : while( $exhibition->have_posts() ) : $exhibition->the_post();
						ovaex_get_template( 'parts/item-exhibition.php', $args_show );
					endwhile; endif; wp_reset_postdata(); wp_reset_query(); ?>
				</div>   

			</div>

			<?php if($show_loadmore == 'yes') : ?>
				<div class="ova_more_post" 
					data-behav="load" 
					data-type_action="<?php echo esc_attr( $default_filter ); ?>" 
					data-paged="2" 
					data-post="exhibition" 
					data-perpage="<?php echo esc_attr( $ovaex_post_per_page ); ?>" 
					data-order-by="<?php echo esc_attr( $ovaex_order_by ); ?>" 
					data-order="<?php echo esc_attr( $ovaex_order ); ?>" 
					data-upcoming="<?php echo esc_attr( $upcoming_day ); ?>" 
					data-categories="<?php echo esc_attr( json_encode( $categories ) ); ?>" 
					data-artist="<?php echo esc_attr( $artist ); ?>"
				>
					<span class="load-more"><?php echo esc_html__('Load More','ovaex'); ?></span>
					<img class="image-loadmore" src="<?php echo OVAEX_PLUGIN_URI ?>/assets/img/loadmore.GIF" alt="loadmore">
				</div>
			<?php endif; ?>

			<div class="ova-nodata"><span><?php echo esc_html__('No Data','ovaex'); ?></span></div>
		</div>
	<?php }
}